#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#ifndef UNITTEST
#include <sys/neutrino.h>
#include "input/mtouch_driver.h"
#include <input/mtouch_log.h>
#endif

#include "touch.h"

/* TODO Erick: Consolidate this with the config/test mode function */
#if 0
int
cypress_operating_mode_display_packet(char *direction, uint8_t reg, uint8_t *packet, int len)
{
  int i;

  mtouch_info (CYPRESS_DEVNAME, "Operating Mode Packet: %s %d", direction, reg);

  for (i = 0; i < len; i++)
    mtouch_info (CYPRESS_DEVNAME, "%x", packet[i]);

  return EOK;
}
#endif

/* TODO Erick: Consolidate this with the config/test mode function */
int
cypress_operating_mode_reply_ready(cypress_dev_t* cypress)
{
    uint8_t data;
    int i;

    if ((cypress->register_map.operating_mode_info == NULL) || (cypress->controller_reset_state == true)) {
        mtouch_error (cypress->log_name, "Controller is resetting; unable to process operating_mode_reply check");
        goto fail;
    }

    for (i = 0; i < cypress->operating_mode_reply_retry_limit; i++) {
        if ((cypress->i2c_funcs.read_reg(cypress->i2c_fd, cypress->register_map.operating_mode_info->cmd_ofs, sizeof(data), &data)) < 0) {
            mtouch_error (cypress->log_name, "Failed to read touch controller status");
            goto fail;
        }

        if (data & COMMAND_COMPLETE) {
            if (cypress->verbose > 5)
                mtouch_info (cypress->log_name, "Controller has processed command, response is waiting");

            return EOK;
        }

        delay (cypress->operating_mode_reply_retry_delay);
    }

    mtouch_warn (cypress->log_name, "Controller took to long too reply to Operating Mode command");

fail:

    return -1;
}

/* TODO Erick: Consolidate this with the config/test mode function */
int
cypress_read_operating_mode_reply(cypress_dev_t* cypress, uint8_t **reply, int8_t len)
{
    uint8_t *buf = 0;

    if (cypress->controller_reset_state == true) {
        mtouch_error (cypress->log_name, "Controller is resetting; unable to process read_operating_mode_reply");
        return -1;
    }

    /* Some commands we don't know the reply length */
    if (len == -1) {
        /* Read length data, this appears to vary depending on the command */
        mtouch_error (cypress->log_name, "Invalid length");
        return -1;
    }

    buf =(uint8_t *) calloc(len, sizeof(uint8_t));
    if ((cypress->i2c_funcs.read_reg(cypress->i2c_fd, (cypress->register_map.operating_mode_info->cmd_ofs + CMD_DATA), len, buf)) < 0) {
        mtouch_error (cypress->log_name, "Failed to read touch controller status");
        return -1;
    }

    if (*reply != NULL)
        free (*reply);

    *reply =(uint8_t *) calloc (len, sizeof (uint8_t));

    if (*reply == NULL) {
        mtouch_error (cypress->log_name, "Failed to allocate memory for configuration data response");
        return -1;
    }

    memcpy (*reply, buf, len);

    if (cypress->verbose > 5)
    //	cypress_operating_mode_display_packet("r", (cypress->register_map.operating_mode_info->cmd_ofs + CMD_DATA), *reply, len);
        cypress_display_packet(cypress, "r", (cypress->register_map.operating_mode_info->cmd_ofs + CMD_DATA), *reply, len);

    free(buf);
    return len;
}

int
cypress_operating_mode_cmd(cypress_dev_t* cypress, uint8_t cmd, uint8_t *data, uint8_t data_len, uint8_t **reply, uint8_t reply_len)
{
    int rc = 0;

    if ((cypress->register_map.operating_mode_info == NULL) || (cypress->controller_reset_state == true)) {
        mtouch_error (cypress->log_name, "Controller is resetting; unable to process operating_mode_cmd");
        return -1;
    }

    if (data != NULL) {
        /* We write the data parameters first */
        rc = cypress->i2c_funcs.write_reg(cypress->i2c_fd, (cypress->register_map.operating_mode_info->cmd_ofs + CMD_DATA), data_len, data);
        if (0 != rc) {
            mtouch_error (cypress->log_name, "Failed to write command parameter");
            return -1;
        }

        if (cypress->verbose > 5)
            //cypress_operating_mode_display_packet("w", (cypress->register_map.operating_mode_info->cmd_ofs + CMD_DATA), data, data_len);
            cypress_display_packet(cypress, "w", (cypress->register_map.operating_mode_info->cmd_ofs + CMD_DATA), data, data_len);
    }

    /* Ensure Command_complete is unset */
    cmd &= ~COMMAND_COMPLETE;

    if (cypress->register_map.operating_mode_info == NULL)
        return -1;

    /* Write the command and the COMMAND_COMPLETE bit, this tells the controller 'go!' */
    rc = cypress->i2c_funcs.write_reg(cypress->i2c_fd, cypress->register_map.operating_mode_info->cmd_ofs, sizeof(cmd), &cmd);
    if (0 != rc) {
        mtouch_error (cypress->log_name, "Failed to send Config_Test command %x", cmd);
        return -1;
    }

    if (cypress->verbose > 5)
        //cypress_operating_mode_display_packet("w", cypress->register_map.operating_mode_info->cmd_ofs, &cmd, sizeof(cmd));
        cypress_display_packet(cypress, "w", cypress->register_map.operating_mode_info->cmd_ofs, &cmd, sizeof(cmd));

    if (EOK != cypress_operating_mode_reply_ready(cypress)) {
        mtouch_error (cypress->log_name, "Operating Mode command %x failed", cmd);
        return -1;
    }

    switch ((cmd)) {
    case CYPRESS_OPERATING_MODE_PING:
        /* Nothing to be done, just return */
        break;
    case CYPRESS_OPERATING_MODE_GET_PARAMETER:
    case CYPRESS_OPERATING_MODE_SET_PARAMETER:
    case CYPRESS_OPERATING_MODE_GET_DATA_BLOCK_CRC:
        /* We don't do anything with these yet */
        /* TODO ERICK: Handle the other commands */
        break;
    case CYPRESS_OPERATING_PLAY_HATPIC_DATA:
	    mtouch_error (cypress->log_name, "read configuration reply");
        if ((rc = cypress_read_operating_mode_reply(cypress, reply, reply_len))) {
            if (-1 == rc) {
                mtouch_error (cypress->log_name, "Failed to read configuration reply");
                return -1;
            }
        }
        break;
    default:
        mtouch_warn (cypress->log_name, "Unknown Operating Mode Command %x", cmd);
        break;
    }

    return EOK;
}

int
cypress_operating_mode_heartbeat(cypress_dev_t* cypress)
{
    int rc = 0;

    /* Check the current mode and ensure we are in Operating Mode */
    switch (cypress_get_mode (cypress)) {
    case BOOTLOADER_MODE:
            /* ESD might have put us back into bootloader mode. Reset and start again. */
    mtouch_warn (cypress->log_name, "Heartbeat, Controller is unexpectedly back in bootloader mode, resetting the controller and driver");
            cypress_controller_reset (cypress);
    rc = -1;
            goto fail;
            break;
    case OPERATING_MODE:
            /* Send Ping */
    if (cypress->verbose > 4)
        mtouch_info (cypress->log_name, "Sending ping");

    if (-1 == (cypress_operating_mode_cmd(cypress, CYPRESS_OPERATING_MODE_PING, NULL, 0, NULL, 0))) 
    {
        /* No reply */
        mtouch_warn (cypress->log_name, "Ping command failed, resetting the controller and driver");
        error_memory("Cypress_Touch: Ping command failed, resetting the controller and driver");
        rc = -1;
        // goto fail;
    }
    else {
        if (cypress->verbose > 6) {
        mtouch_info (cypress->log_name, "Heartbeat pulse");
        }
    }

    if (cypress->controller_reset_count < 2)
    {
        mtouch_error(cypress->log_name, "Resetting the controller due to hearbeat supervision failure");
        /* release any previous press touch events */
        cypress_release_touch(cypress);
        if (EOK != cypress_controller_reset(cypress)) {
            mtouch_error(cypress->log_name, "cypress_operating_mode_heartbeat:Failed to reset controller");
        }
        else {
            cypress->controller_reset_count++;
        }
    } else {
        mtouch_warn(cypress->log_name, "cypress_operating_mode_heartbeat:Maximum controller reset attempts reached for RTD supervision");
    }
    break;
    default:
        /* Anything else is wrong, reset the controller */
        mtouch_error(cypress->log_name, "Resetting the controller due to hearbeat supervision failure, entered default case");
        cypress_controller_reset (cypress);
        rc = -1;
        goto fail;
        break;
    }

    /* (Re)Start timer */
    if (EOK != cypress_set_heartbeat_timer(cypress, cypress->heartbeat_sleep)) {
        mtouch_error (cypress->log_name, "%s failed to set the heartbeat timer", __FUNCTION__);
        rc = -1;
    }

fail:
    return rc;
}

int cypress_play_haptic_data(cypress_dev_t* cd, uint8_t *haptic_effect)
{
    uint8_t *reply = NULL;

    if(haptic_effect == NULL)
        return -EINVAL;

    mtouch_debug(cd->log_name, "Haptic Sound Effect:0x%02X \nHaptic Vibration Effect:0x%02X", (*haptic_effect & 0xF0) >> 4, (*haptic_effect & 0x0F));
    pthread_mutex_lock(&cd->ctrl_reset_mutex);
    if(cd->controller_reset_state == false)
    {
        if (-1 == (cypress_operating_mode_cmd(cd, CYPRESS_OPERATING_PLAY_HATPIC_DATA, haptic_effect, 1, &reply, CYPRESS_OPERATING_PLAY_HAPTIC_RET_SZ))) {
            mtouch_warn (cd->log_name, "Failed to send play haptic effect command");
            pthread_mutex_unlock(&cd->ctrl_reset_mutex);
            return -1;;
        }
    }
    else
    {
      mtouch_warn (cd->log_name, "Controller is undergoing reset, hence cannot perform Play haptic");
      pthread_mutex_unlock(&cd->ctrl_reset_mutex);
      return -1;
    }
    pthread_mutex_unlock(&cd->ctrl_reset_mutex);
    if (reply == NULL) {
        mtouch_info (cd->log_name, "Send play haptic command failed");
        return -1;
    }

    if (!*reply)
        mtouch_info (cd->log_name, "Set play haptic data successful %d", *reply);
    else
        mtouch_error (cd->log_name, "Failed to set play haptic command %d", *reply);

    if (*reply)
        *reply = -1;

    return *reply;
}

/* NOTE: In order for the driver to properly obtain the noise levels the hardware must have the functionality enabled in the firmware */
int
cypress_operating_mode_fetch_noise_levels(cypress_dev_t* cypress, uint16_t *noise_level, uint16_t *noise_level_velocity)
{
    uint8_t *buf;
    int noise_level_offset = 0;
    int noise_level_len = 0;
    int noise_level_velocity_offset = 0;
    int noise_level_velocity_len = 0;

    *noise_level = 0;
    *noise_level_velocity = 0;

    switch (cypress->silicon_gen) {
    case GEN_FOUR:
        noise_level_offset = ((cypress->register_map.operating_mode_info->rep_ofs + 0x03) + (abs (cypress->register_map.operating_mode_info->num_btns / 4)) +
                              ((cypress->register_map.operating_mode_info->obj_cfg0 == 1) ? 2: 0));
        noise_level_len = 2;

        noise_level_velocity_offset = (noise_level_offset + 0x02);
        noise_level_velocity_len = noise_level_len;
        break;
    case GEN_SIX:
        noise_level_offset = ((cypress->register_map.operating_mode_info->rep_ofs + 4) + (abs (cypress->register_map.operating_mode_info->num_btns / 4)) + 8);
        noise_level_len = 1;

        /* Noise level velocity is unsupported in Gen 6 controllers */
        break;
    }

    buf =(uint8_t *) calloc(noise_level_len, sizeof(uint8_t));
    /* Read the noise levels */
    if ((cypress->i2c_funcs.read_reg(cypress->i2c_fd, noise_level_offset, noise_level_len, buf)) < 0) {
        mtouch_error (cypress->log_name, "Failed to read Noise Levels.");
        return -1;
    }

    memcpy (noise_level, buf, noise_level_len);

    /* Read the noise level velocity (if supported) */
    if (noise_level_velocity_offset) {
        if ((cypress->i2c_funcs.read_reg(cypress->i2c_fd, noise_level_offset, noise_level_len, buf)) < 0) {
            mtouch_error (cypress->log_name, "Failed to read Noise Levels.");
            return -1;
        }

        memcpy (noise_level_velocity, buf, noise_level_velocity_len);
    }

    return EOK;
}

#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn.ott.qnx.com/product/branches/7.0.0/trunk/hardware/mtouch/cypress/operating_mode.c $ $Rev: 871068 $")
#endif
